"use client";

import React, { useState, useEffect } from "react";
import InnerBanner from "../../../../components/InnerBanner";
import "../../page.css";
import Link from "next/link";
import { useCart } from "@/context/CartContext";

function stripHtml(html: string): string {
    if (typeof window !== "undefined") {
        const div = document.createElement("div");
        div.innerHTML = html;
        return div.textContent || div.innerText || "";
    }
    return html.replace(/<[^>]*>?/gm, "").trim();
}

type SelectionState = {
    type?: string;
    medium?: string;
    session?: string;
};

interface SolvedCategoryClientProps {
    courseData: any;
    assignmentsList: any[];
    courseCode: string;
}

export default function SolvedCategoryClient({ courseData, assignmentsList, courseCode }: SolvedCategoryClientProps) {
    const [mounted, setMounted] = useState(false);
    const [search, setSearch] = useState("");

    useEffect(() => {
        setMounted(true);
    }, []);

    const { addToCart } = useCart();
    const [selections, setSelections] = useState<Record<string, SelectionState>>({});
    const [activeCartSlug, setActiveCartSlug] = useState<string | null>(null);

    const headline = courseData?.course_headline || "";
    const breadcrumbHeadline = courseData?.breadcrumb_headline || "";
    const description = courseData?.description ? stripHtml(courseData.description) : "";

    const updateSelection = (
        slug: string,
        field: keyof SelectionState,
        value: string
    ) => {
        setSelections(prev => ({
            ...prev,
            [slug]: {
                ...prev[slug],
                [field]: value,
            },
        }));
    };

    const handleAddToCart = async (item: any) => {
        const selection = selections[item.assignment_slug];
        if (!selection) return;

        setActiveCartSlug(item.assignment_slug);
        const rawString = `${courseCode}|${item.assignment_slug}|${Date.now()}`;

        try {
            await addToCart({
                id: `solved-assignment-${rawString}`,
                name: item.title ?? "",
                price: item.sale ?? 0,
                qty: 1,
                img: item.img ?? "",
                courseCode,
                itemCode: item.assignment_slug,
                subjectIds: null,
                selectedType: selection.type!,
                selectedMedium: selection.medium!,
                selectedSession: selection.session!,
                dataSaveId: null,
                pageName: "solved-assignment-page",
            });
        } finally {
            setActiveCartSlug(null);
        }
    };

    const filteredAssignments = assignmentsList.filter(a =>
        a.title.toLowerCase().includes(search.toLowerCase()) ||
        a.code?.toLowerCase().includes(search.toLowerCase())
    );

    if (!mounted) return null;

    return (
        <>
            <InnerBanner title={headline} breadcrumb={breadcrumbHeadline} />

            <div className="page-container">
                <div className="container">
                    <div className="d-lg-flex d-md-block d-block align-items-center gap-2 mb-4 justify-content-between">
                        <div className='d-flex '>
                            <h2 className="fw-bold">
                                {headline}
                            </h2>
                            <p dangerouslySetInnerHTML={{ __html: courseData?.description || "" }} />
                        </div>

                        {/* 🔥 Search Box */}
                        <div className="search-box mb-4 w-50 position-relative">
                            <input
                                type="text"
                                className="form-control "
                                placeholder="Search by Title or Assignment Code..."
                                value={search}
                                onChange={(e) => setSearch(e.target.value)}
                            />

                            <i className="fa-solid fa-magnifying-glass"></i>
                        </div>
                    </div>

                    <div className="row g-4">
                        {filteredAssignments.length > 0 ? (
                            filteredAssignments.map((item: any, i: number) => {
                                const discount = Math.round(((item.price - item.sale) / item.price) * 100);

                                const currentSelection = selections[item.assignment_slug];
                                const isDisabled =
                                    !currentSelection?.type ||
                                    !currentSelection?.medium ||
                                    !currentSelection?.session ||
                                    activeCartSlug === item.assignment_slug;

                                return (
                                    <div key={i} className="col-12 col-sm-6 col-md-6 col-lg-4 col-xl-3">
                                        <div className="card h-100 border-0 shadow-sm product-card">
                                            <div className="position-relative overflow-hidden">
                                                <span className="badge bg-danger position-absolute top-0 end-0 m-3 z-2">
                                                    Sale!
                                                </span>
                                                <div className="product-image-wrapper m-0">
                                                    <img
                                                        src={item.img}
                                                        className="card-img-top product-image"
                                                        alt={item.title}
                                                    />
                                                </div>
                                            </div>

                                            <div className="card-body p-3">
                                                <Link href={`/assignments/solved/${courseCode}/${item.assignment_slug}`} className="text-decoration-none">
                                                    <h5 className="card-title text-primary fw-bold mb-1">{item.code}</h5>
                                                </Link>
                                                <p className="card-text text-muted small mb-3 text-truncate" title={item.title}>
                                                    {item.title}
                                                </p>

                                                <div className="mb-3">
                                                    {/* Types */}
                                                    <select
                                                        className="form-select form-select-sm mb-2 shadow-none border-secondary-subtle"
                                                        value={currentSelection?.type || ""}
                                                        onChange={e =>
                                                            updateSelection(item.assignment_slug, "type", e.target.value)
                                                        }
                                                        aria-label="Select Type"
                                                    >
                                                        <option value="">-- Select Type --</option>
                                                        {item.types?.map((type: { enc_id: string; type_name: string }) => (
                                                            <option key={type.enc_id} value={type.enc_id}>{type.type_name}</option>
                                                        ))}
                                                    </select>

                                                    {/* Languages */}
                                                    <select
                                                        className="form-select form-select-sm mb-2 shadow-none border-secondary-subtle"
                                                        value={currentSelection?.medium || ""}
                                                        onChange={e =>
                                                            updateSelection(item.assignment_slug, "medium", e.target.value)
                                                        }
                                                        aria-label="Language Selection"
                                                    >
                                                        <option value="">-- Select Medium --</option>
                                                        {item.medium?.map((language: { enc_id: string; language_name: string }) => (
                                                            <option key={language.enc_id} value={language.enc_id}>{language.language_name}</option>
                                                        ))}
                                                    </select>

                                                    {/* Sessions */}
                                                    <select
                                                        className="form-select form-select-sm shadow-none border-secondary-subtle"
                                                        value={currentSelection?.session || ""}
                                                        onChange={e =>
                                                            updateSelection(item.assignment_slug, "session", e.target.value)
                                                        }
                                                        aria-label="Select Session"
                                                    >
                                                        <option value="">-- Select Session --</option>
                                                        {item.sessions?.map((session: { enc_id: string; session_name: string }) => (
                                                            <option key={session.enc_id} value={session.enc_id}>{session.session_name}</option>
                                                        ))}
                                                    </select>

                                                </div>

                                                <div className="d-flex align-items-center mb-3">
                                                    <h4 className="mb-0 fw-bold">₹{item.sale}</h4>
                                                    <span className="text-muted ms-2 text-decoration-line-through small">Only</span>
                                                    <span className="text-muted ms-1 text-decoration-line-through small">MRP ₹{item.price}</span>
                                                    <span className="badge bg-danger ms-auto">{discount}% Off</span>
                                                </div>

                                                <small
                                                    className="text-danger mb-1 d-block"
                                                    style={{ fontSize: "0.72rem", fontStyle: "italic" }}
                                                >
                                                    ** Type, Medium, and Session are mandatory fields.
                                                </small>

                                                <div className="d-grid gap-2">
                                                    <button
                                                        className="btn btn-dark py-2 rounded-3"
                                                        disabled={isDisabled}
                                                        onClick={() => handleAddToCart(item)}
                                                    >
                                                        {activeCartSlug === item.assignment_slug ? (
                                                            <>
                                                                <span className="spinner-border spinner-border-sm" /> Adding...
                                                            </>
                                                        ) : (
                                                            "Add to Cart"
                                                        )}
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                );
                            })
                        ) : (
                            <div className="col-12">
                                <div className="empty-state">
                                    <div className="empty-icon">
                                        <i className="fa-solid fa-folder-open"></i>
                                    </div>
                                    <h3 className="mb-3">No Solved Assignments Available</h3>
                                    <p className="text-muted">
                                        We're currently preparing solved assignments for this category. Please check back soon!
                                    </p>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </>
    );
}
